from dataclasses import dataclass
from pot_libs.sanic_api import Model

from pot_libs.common.components.fields import Cid
from pot_libs.sanic_api.column import Opt, Float, Int, Str, List, Enum
from unify_api.utils.response_code import DbErr, ParamErr, JwtErr, UserErr


@dataclass
class EquipManagementTotalReq(Model):
    '''
    设备管理汇总-请求格式
    '''
    cid: Cid


@dataclass
class EquipManagementListReq(Model):
    '''
    设备管理列表-请求格式
    '''
    cid: Cid
    is_download: int = Int("是否是下载 0-不是 1-是").eg(0)
    page_size: int = Opt(Int("页大小").eg(10))
    page_num: int = Opt(Int("页码").eg(1))


@dataclass
class EquipRunStatusReq(Model):
    '''
    运行统计状态
    '''
    point_id: int = Opt(Int("监测点").eg(260))


@dataclass
class EquipManagementTotalResp(Model):
    '''
    设备管理汇总-返回格式
    '''
    installed_number: int = Int("安装点数").eg(1)
    start_time: str = Str("启用时间").eg("2022-06-16")


@dataclass
class EquipManagementInfo(Model, DbErr, UserErr, JwtErr):
    '''
    设备管理列表详情-返回格式
    '''
    installed_location: str = Opt(Str("安装位置").eg("3#变压器"))
    device_number: str = Opt(Str("设备编号").eg("A1904000083"))
    wiring_type: str = Opt(Str("接线形式").eg("三表法"))
    ct_change: str = Opt(Int("ct变比").eg(1000))
    pt_change: str = Opt(Int("pt变比").eg(1000))
    rated_voltage: int = Opt(Int("额定电压").eg(400))
    start_time: str = Str("接线时间").eg("2022-06-17 16:42")


@dataclass
class EquipManagementListResp(Model, DbErr, UserErr, JwtErr):
    '''
    设备管理列表-返回格式
    '''
    rows: list = List("设备信息").items(EquipManagementInfo)
    total: int = Int("总量")
    page_num: int = Int("当前页").eg(1)


@dataclass
class EquipRunReq(Model):
    '''
    运行统计-请求格式
    '''
    cid: Cid
    page_size: int = Opt(Int("每页记录数").eg(20))
    page_num: int = Opt(Int("当前页码").eg(1))
    is_download: int = Int("是否是下载 0-不是 1-是").eg(0)
    start: str = Opt(Str("开始时间").eg("2020-05-01 00:00:00"))
    end: str = Opt(Str("结束时间").eg("2020-05-01 23:59:59"))
    point_ids: list = Opt(List("监测点 []表示没有 [-1]表示全部").eg([260, 261, 268]))
    sort_field: list = Enum('排序字段').of('point_name', 'start_time', 'end_time',
                                       'run_time')
    sort_type: list = Enum('排序方向').of('asc', 'desc')
    
    @classmethod
    def example(cls):
        return {
            "cid": 154,
            "point_ids": [
                182,
                421,
                422,
                423,
                1752,
                1753,
                1754
            ],
            "start": "2022-06-28 00:00:00",
            "end": "2022-06-28 23:59:59",
            "is_download": 0,
            "page_num": 1,
            "page_size": 20,
            "sort_field": "start_time",
            "sort_type": "desc"
        }


@dataclass
class EquipRunInfo(Model, DbErr, UserErr, JwtErr):
    '''
    运行统计列表详情-返回格式
    '''
    point_name: str = Opt(Str("监测点").eg("3#变压器"))
    start_time: str = Str("开启时间").eg("2022-06-17 16:42")
    end_time: str = Str("关闭时间").eg("2022-06-21 16:42")
    run_time: str = Str("运行时长").eg("1小时20分")


@dataclass
class EquipRunListResp(Model, DbErr, UserErr, ParamErr):
    '''
    运行统计-返回格式
    '''
    rows: list = List("运行信息").items(EquipRunInfo)
    total: int = Int("总量")
    page_num: int = Int("当前页").eg(1)


@dataclass
class EquipRunStatisticsReq(Model):
    '''
    运行统计数据-请求格式
    '''
    cid: Cid
    start: str = Opt(Str("开始时间").eg("2020-05-01 00:00:00"))
    end: str = Opt(Str("结束时间").eg("2020-05-01 23:59:59"))
    point_ids: list = Opt(List("监测点 []表示没有 [-1]表示全部").eg([260, 261, 268]))
    
    @classmethod
    def example(cls):
        return {
            "cid": 84,
            "point_ids": [
                261,
                262,
                266,
                268
            ],
            "start": "2021-09-06 15:04:51",
            "end": "2021-12-06 15:04:51",
        }


@dataclass
class EquipRunStatisticsResp(Model, ParamErr):
    '''
    运行统计数据-返回格式
    '''
    count: int = Int("运行次数").eg(23)
    run_all_time: str = Str("运行时长").eg("11小时21分")
    run_avg_time: str = Str("平均时长").eg("1小时05分")
    run_max_time: str = Str("最长时长").eg("2小时23分")


@dataclass
class EquipRunStatusResp(Model, ParamErr):
    '''
    运行统计状态-返回格式
    '''
    is_run: int = Int("是否运行:0-否、1-是 2-非动力设备").eg(1)
