from dataclasses import dataclass

from pot_libs.common.components.fields import Cid, Item
from pot_libs.sanic_api import Model
from pot_libs.sanic_api.column import Int, List, Opt, Str, Float, Dict, Enum
from unify_api.utils.response_code import DbErr, ServerErr, ParamErr


@dataclass
class ScopeListReq(Model):
    '''
    录波识别记录-请求格式
    '''
    cid: Cid
    page_size: int = Opt(Int("每页记录数").eg(20))
    page_num: int = Opt(Int("当前页码").eg(1))
    is_download: int = Int("是否是下载 0-不是 1-是").eg(0)
    start: str = Opt(Str("开始时间").eg("2022-06-22 00:00:00"))
    end: str = Opt(Str("结束时间").eg("2022-06-22 23:59:59"))
    scope_g: str = Opt(List("录波颗粒度").eg(["2s"]))
    pids: list = Opt(List("检测点 []表示没有 [-1]表示全部").eg([260, 261, 268]))


@dataclass
class ScopeListDownloadReq(Model):
    '''
    2s录波识记录下载-请求格式
    '''
    start: str = Opt(Str("开始时间").eg("2022-06-22 00:00:00"))
    end: str = Opt(Str("结束时间").eg("2022-06-22 23:59:59"))
    pids: list = Opt(List("监测点 []表示没有 [-1]表示全部").eg([260, 261, 268]))


@dataclass
class ScopeListItem(Model):
    '''
    录播识别记录-每一条数据
    '''
    check_dt: str = Str("触发时间").eg("2021-07-01 23:59:59")
    point: str = Str('监测点名称').eg('土建区总进线')
    message: str = Str('触发原因').eg('B相电压波动')
    scope_type: int = Opt(Int('录波颗粒度').eg('0.25ms'))
    doc_id: str = Opt(Str('该条信息id').eg('213_over_gap_i__1604302769'))


@dataclass
class ScopeListResp(Model):
    '''
    录波识别记录-返回格式
    '''
    total: int = Opt(Int("总条数").eg(20))
    rows: list = Opt(List("总数据").items(ScopeListItem))
    page_num: int = Int("当前页").eg(1)


@dataclass
class ScopeItemDownload(Model):
    """
        录波下载数据详情
    """
    datetime: str = Str("时间").eg("2021-07-01 23:59:59")
    ua: float = Float('A相电压')
    ub: float = Float('B相电压')
    uc: float = Float('C相电压')
    ia: float = Float('A相电流')
    ib: float = Float('B相电流')
    ic: float = Float('C相电流')
    pttl: float = Float('功率')
    lc: float = Float('漏电流')


@dataclass
class ScopeListDownloadResp(Model):
    """
        录波下载数据
    """
    rows: list = Opt(List("数据列表").items(ScopeItemDownload))


@dataclass
class ScopeContent(Model):
    '''
    识别详情图表-返回格式
    '''
    item: Item
    value_datas: list = List().items(Float())


@dataclass
class ScopeDetailRep(Model):
    '''
    识别详情-请求格式
    '''
    id: str = Opt(Str("doc_id").eg("423_over_gap_i__1655991008"))

    @classmethod
    def example(cls):
        return {
            "id": "423_over_gap_i__1655991008"
        }


@dataclass
class ScopeDetailsResp(Model, DbErr):
    '''
    识别详情-返回格式
    '''
    point: str = Str('监测点名称').eg('土建区总进线')
    contin_time: int = Opt(Int('持续时间').eg(2))
    scope_type: str = Str('录波颗粒度').eg('0.25ms')
    check_dt: str = Str("发生时间").eg("2021-07-01 23:59:59")
    ctnum: str = Str("接线法 2-两表法 3-三表法").eg(3)
    location: int = Opt(Int('位置').eg(240))
    item: str = Item
    type: str = Opt(Str().eg('电流波动'))
    i: list = List("电流事件录波").items(ScopeContent)
    v: list = List("电压事件录波").items(ScopeContent)
    residual_current: list = Opt(List("2s录波漏电流").items(ScopeContent))
    p: list = Opt(List("2s录波功率").items(ScopeContent))

    @classmethod
    def example(cls):
        return {
            "point": "钻孔配电柜",
            "contin_time": "400",
            "check_dt": "2022-06-23 21:30:08",
            "scope_type": "0.25ms",
            "item": "A相",
            "type": "电流波动",
            "i": [{
                "item": "ia",
                "value_datas": [1.25, -0.34]
            }],
            "v": [{
                "item": "ua",
                "value_datas": [86.06, 123.34]
            }],
            "residual_current": [{
                "item": "漏电流",
                "value_datas": [86.06, 123.34]
            }],
            "power": [{
                "item": "p",
                "value_datas": [86.06, 123.34]
            }],
        }


@dataclass
class GetScopeConfigReq(Model):
    '''
    获取录播配置-请求格式
    '''
    pid: int = Int("监测点id").eg(1754)


@dataclass
class SetScopeConfigReq(Model):
    '''
    设置录播配置-请求格式
    '''
    pid: int = Int("监测点id").eg(20)
    scope_type: str = Enum("录波颗粒度").of('0.25ms', '0.2s', '2s')
    type: str = Enum("提交类型").of("state", "i", "v", "residual_current", "power",
                                "time")
    state: int = Opt(Enum("开启关闭状态").of(0, 1))
    umax: int = Opt(Int("电压上限").eg(430))
    umin: int = Opt(Int("电压下限").eg(360))
    ugap: int = Opt(Int("电压波动阈值").eg(50))
    imax: int = Opt(Int("电流上限").eg(40))
    igap: int = Opt(Int("电流波动阈值").eg(5))
    lcmax: int = Opt(Int("漏电流上限").eg(40))
    lcgap: int = Opt(Int("漏电流波动阈值").eg(40))
    pttlmax: int = Opt(Int("功率上限").eg(5))
    pttlgap: int = Opt(Int("功率波动阈值").eg(40))
    one_time: list = Opt(List("第一段时间").items(Str()))
    two_time: list = Opt(List("第二段时间").items(Str()))
    three_time: list = Opt(List("第三段时间").items(Str()))


@dataclass
class GetScopeConfigResp(Model, ServerErr):
    '''
    获取录播配置-返回格式
    '''
    pid: int = Int("监测点id").eg(20)
    rows: list = List("配置内容").items(SetScopeConfigReq)

    @classmethod
    def example(cls):
        return {
            "pid": 1754,
            "rows": {
                "0.25ms": {
                    "state": 1,
                    "configs": {
                        "umin": {
                            "name": "越下限阈值",
                            "type": "v",
                            "value": 85
                        },
                        "umax": {
                            "name": "越上限阈值",
                            "type": "v",
                            "value": 115
                        },
                        "ugap": {
                            "name": "波动阈值",
                            "type": "v",
                            "value": 25
                        },
                        "imax": {
                            "name": "越限阈值",
                            "type": "i",
                            "value": 5
                        },
                        "igap": {
                            "name": "波动阈值",
                            "type": "i",
                            "value": 20
                        }
                    }
                },
                "0.2s": {
                    "state": 1,
                    "configs": {
                        "umin": {
                            "name": "越下限阈值",
                            "type": "v",
                            "value": 187
                        },
                        "umax": {
                            "name": "越上限阈值",
                            "type": "v",
                            "value": 253
                        },
                        "ugap": {
                            "name": "波动阈值",
                            "type": "v",
                            "value": 50
                        },
                        "imax": {
                            "name": "越限阈值",
                            "type": "i",
                            "value": 5
                        },
                        "igap": {
                            "name": "波动阈值",
                            "type": "i",
                            "value": 1
                        }
                    }
                },
                "2s": {
                    "state": 1,
                    "configs": {
                        "umin": {
                            "name": "越下限阈值",
                            "type": "v",
                            "value": 187
                        },
                        "umax": {
                            "name": "越上限阈值",
                            "type": "v",
                            "value": 253
                        },
                        "ugap": {
                            "name": "波动阈值",
                            "type": "v",
                            "value": 50
                        },
                        "imax": {
                            "name": "越限阈值",
                            "type": "i",
                            "value": 5
                        },
                        "igap": {
                            "name": "波动阈值",
                            "type": "i",
                            "value": 1
                        },
                        "lcmax": {
                            "name": "越限阈值",
                            "type": "residual_current",
                            "value": 1
                        },
                        "lcgap": {
                            "name": "波动阈值",
                            "type": "residual_current",
                            "value": 1
                        },
                        "pttlmax": {
                            "name": "越限阈值",
                            "type": "power",
                            "value": 5
                        },
                        "pttlgap": {
                            "name": "波动阈值",
                            "type": "power",
                            "value": 2
                        },
                        "one_time": {
                            "name": "第一个时间段",
                            "type": "time",
                            "value": [
                                "08:00",
                                "08:30"
                            ]
                        },
                        "two_time": {
                            "name": "第二个时间段",
                            "type": "time",
                            "value": [
                                "09:00",
                                "10:00"
                            ]
                        },
                        "three_time": {
                            "name": "第三个时间段",
                            "type": "time",
                            "value": [
                                "14:00",
                                "15:00"
                            ]
                        }
                    }
                }
            }
        }


@dataclass
class InitScopeConfigReq(Model):
    '''
        初始化配置信息--请求
    '''
    pids: list = List('pids')

    @classmethod
    def example(cls):
        return {
            'pids': [238, 240, 242, 330, 343, 749, 1463, 2248]
        }


@dataclass
class FlushScopeEsDataReq(Model):
    '''
        刷新es录波数据--请求
    '''
    scope_type_list: list = List('录波类型')
    start_time: str = Str('开始时间')
    end_time: str = Str('结束时间')

    @classmethod
    def example(cls):
        return {
            'scope_type_list': ['200ms', '2s', '0.25ms'],
            'start_time': '2022-07-04 17:40:24',
            'end_time': '2022-07-21 18:40:24',
        }


@dataclass
class GetScopeConfigList(Model):
    '''
    获取录播配置列表
    '''
    state: int = Enum("状态 0-关 1-开").of(0, 1)
    configs: list = List("返回配置")


@dataclass
class SetScopeConfigResp(Model):
    '''
    设置录播配置-返回格式
    '''
    success: int = Int('是否操作成功 0-否 1-是').eg(1)
    message: str = Str("返回信息").eg("操作成功")


# 识别记录列表请求举例
scope_list_req_example = {
    "范例1": {
        "cid": 114,
        "page_size": 20,
        "page_num": 1,
        "is_download": 0,
        "start": "2022-06-22 00:00:00",
        "end": "2022-06-22 23:59:59",
        "pids": []
    },
    "范例2": {
        "cid": 44,
        "page_size": 20,
        "page_num": 1,
        "is_download": 0,
        "start": "2022-06-22 00:00:00",
        "end": "2022-06-22 23:59:59",
        "pids": [260, 261, 268]
    }

}

'''
设置录波配置格式
'''
set_scope_config_example = {
    "0.25ms设置电压范例": {
        "pid": 1754,
        "scope_type": "0.25ms",
        "type": "v",
        "umax": 430,
        "umin": 360,
        "ugap": 50,
    },
    "0.25ms设置状态范例": {
        "pid": 1754,
        "scope_type": "0.25ms",
        "type": "state",
        "state": 1,
    },
    "0.2s设置电流范例": {
        "pid": 1754,
        "scope_type": "0.2s",
        "type": "i",
        "imax": 40,
        "igap": 5,
    },
    "2s设置漏电流范例": {
        "pid": 1754,
        "scope_type": "2s",
        "type": "residual_current",
        "lcmax": 30,
        "lcgap": 12
    },
    "2s设置功率范例": {
        "pid": 1754,
        "scope_type": "2s",
        "type": "power",
        "pttlmax": 5,
        "pttlgap": 40,
    },
    "2s设置时间段范例": {
        "pid": 1754,
        "scope_type": "2s",
        "type": "time",
        "one_time": ['08:00', '09:00'],
        "two_time": ['10:00', '10:30'],
        "three_time": ['13:00', '14:00'],
    },
}

'''
    初始化录播的配置
'''
init_scope_config_example = {
    "coef": {
        "Kua": 2.843364,
        "Bua": 2070.924316,
        "Kub": 2.849909,
        "Bub": 2058.210205,
        "Kuc": 2.854909,
        "Buc": 2054.158447,
        "Kia": 9.65814,
        "Bia": 2056.642578,
        "Kib": 9.652307,
        "Bib": 2054.431152,
        "Kic": 9.65714,
        "Bic": 2056.80249
    },
    "threshold": {
        "0.25ms": {
            "en_scope": 0,
            "umin": 187,
            "umax": 253,
            "ugap": 11.5,
            "igap": 8,
            "imax": 167,
        },
        "0.2s": {
            "en_wave_200ms": 0,
            "umin": 187,
            "umax": 253,
            "ugap": 11.5,
            "igap": 8,
            "imax": 167,
            "scop_limit": 50
        },
        "2s": {
            "en_electric_2s": 0,
            "start_time_I": "11:00:00",
            "stop_time_I": "12:00:00",
            "start_time_II": "14:00:00",
            "stop_time_II": "15:00:00",
            "start_time_III": "15:00:00",
            "stop_time_III": "16:00:00",
            "umin": 187,
            "umax": 253,
            "ugap": 11.5,
            "igap": 15,
            "imax": 167,
            "lcmax": 30,
            "lcgap": 12,
            "pttlgap": 10,
            "pttlmax": 110
        }
    }
}


@dataclass
class ScopeDetail(Model):
    name: str = Str("名称").eg("ia")
    value: list = List("值")


@dataclass
class ScopeDetails(Model):
    item: str = Str("名称").eg("ia")
    value_datas: list = List("值")


@dataclass
class ScopeDetailResp(Model):
    point: str = Str("监测点名称").eg("华侨新村270栋101")
    ctnum: int = Int("接线方式 2-两表法 3-三表法").eg(1)
    sid: str = Str("硬件编号").eg("A2270457094")
    check_dt: str = Str("触发时间").eg("2022-05-26 15:55:03")
    contin_time: str = Str("录波时长").eg("400ms")
    item: str = Str("相限").eg("A相")
    scope_g: str = Str("录波颗粒度 0.25ms/0.2s/2s").eg("0.25ms")
    type: str = Str("录波类型").eg("over_res_cur")
    location: int = Int("location触发点").eg(0)
    v: list = List("电压曲线").items(ScopeDetails)

    i: list = List("电流曲线").items(ScopeDetails)

    residual_current: list = List("漏电流曲线").items(ScopeDetails)

    p: list = List("功率曲线").items(ScopeDetails)

