# -*- coding:utf-8 -*-
"""
DATE：2024/12/30 10:16

"""
from gurobipy import Model, GRB


class EnergyModel:
    def __init__(self, d_params):
        self.load_year = d_params["load_year"]
        self.imp_load_year = d_params["imp_load_year"]
        self.cg_cost = d_params["cg_cost"]
        self.es_cost = d_params["es_cost"]
        self.cg_eff = d_params["cg_eff"]
        self.es_eff = d_params["es_eff"]
        self.work_hours_year = d_params["work_hours_year"]
        self.ele_price = d_params["ele_price"]
        self.max_invest_year = d_params["max_invest_year"]

    def cri_calculate(self):
        m = Model("Energy_Investment_Optimization")
        P_ES = m.addVar(vtype=GRB.CONTINUOUS, name="P_ES")
        P_CG = m.addVar(vtype=GRB.CONTINUOUS, name="P_CG")
        IRR = m.addVar(vtype=GRB.CONTINUOUS, name="IRR")

        E_CG = P_CG * self.cg_eff * self.work_hours_year
        Revenue = self.ele_price * E_CG
        Investment_Cost = P_ES * self.es_cost + P_CG * self.cg_cost

        m.setObjective(IRR, GRB.MINIMIZE)
        m.addConstr(IRR * Revenue == Investment_Cost,
                    "Critical_Load_Constraint")
        m.addConstr(P_CG * self.cg_eff >= self.imp_load_year,
                    "Critical_Load_Constraint")
        m.addConstr(P_ES * self.es_eff >= self.imp_load_year,
                    "Load_Constraint")
        m.addConstr(P_ES * self.es_eff <= self.imp_load_year * 1.2,
                    "ES_Capacity_Constraint")
        m.addConstr(P_CG * self.cg_eff <= self.imp_load_year * 1.2,
                    "CG_Capacity_Constraint")

        m.optimize()

        if m.status == GRB.OPTIMAL:
            irr = Investment_Cost.getValue() / Revenue.getValue()
            if irr > self.max_invest_year:
                return {"Error": f"当前区域的投资回报比（IRR = {irr:.2f}）高于"
                                 f"设置的投资上限，建议不要投资。"}
            else:
                result = {
                    'pv': 0,
                    'es': P_ES.x,
                    'cg': P_CG.x,
                    'income_year': Revenue.getValue(),
                    'total_inv_cost': Investment_Cost.getValue(),
                    'irr': round(irr, 2)
                }
                return result
        else:
            return {"Error": "当前求解的限制条件不合理"}

    def all_calculate(self, d_params):
        pv_cost = d_params["pv_cost"]
        sun_hrs_year = d_params["sun_hrs_year"]
        max_load = d_params["max_load"]
        min_load = d_params["min_load"]
        pv_eff = d_params["pv_eff"]
        pv_area = d_params["pv_area"]
        pv_limit = d_params["pv_limit"]

        pv_area_ele = 0.15

        m = Model("Energy_Investment_Optimization")
        P_PV = m.addVar(vtype=GRB.CONTINUOUS, name="P_PV")
        P_ES = m.addVar(vtype=GRB.CONTINUOUS, name="P_ES")
        P_CG = m.addVar(vtype=GRB.CONTINUOUS, name="P_CG")
        IRR = m.addVar(vtype=GRB.CONTINUOUS, name="IRR")
        E_PV = P_PV * sun_hrs_year * pv_eff

        E_CG = P_CG * self.work_hours_year * self.cg_eff

        Revenue = self.ele_price * self.load_year * self.work_hours_year
        Investment_Cost = P_PV * pv_cost + P_ES * self.es_cost + P_CG * self.cg_cost
        m.setObjective(IRR, GRB.MINIMIZE)
        m.addConstr(E_PV + E_CG >= self.load_year * self.work_hours_year,
                    "IRR_Constraint")

        m.addConstr(IRR * Revenue == Investment_Cost, "IRR_Constraint")
        m.addConstr(P_PV * pv_eff + E_CG * self.cg_eff >= max_load,
                    "PV_Constraint")
        m.addConstr(P_PV <= min(pv_area_ele * pv_area, pv_limit),
                    "PV_Limit_Constraint")
        m.addConstr(P_CG * self.cg_eff >= self.imp_load_year, "CG_Constraint")
        m.addConstr(P_CG * self.cg_eff + P_ES * self.es_eff >= min_load,
                    "CG_Min_Load_Constraint")

        m.addConstr(P_ES * self.es_eff >= self.imp_load_year,
                    "ES_Load_Constraint")
        m.addConstr(P_ES * self.es_eff >= 0.2 * pv_eff * P_PV,
                    "ES_PV_Constraint")

        m.optimize()

        if m.status == GRB.OPTIMAL:
            irr = IRR.x
            if irr > self.max_invest_year:
                return {"Error": f"当前区域的投资回报比（IRR = {irr:.2f}）高于"
                                 f"设置的投资上限，建议不要投资。"}
            else:
                result = {
                    'pv': P_PV.x,
                    'es': P_ES.x,
                    'cg': P_CG.x,
                    'income_year': Revenue,
                    'total_inv_cost': Investment_Cost.getValue(),
                    'irr': round(irr, 2)
                }
                return result
        else:
            return {"Error": "当前求解的限制条件不合理"}

    def sale_calculate(self, d_params):
        pv_cost = d_params["pv_cost"]
        sun_hrs_year = d_params["sun_hrs_year"]
        max_load = d_params["max_load"]
        min_load = d_params["min_load"]
        pv_eff = d_params["pv_eff"]
        pv_area = d_params["pv_area"]
        pv_limit = d_params["pv_limit"]

        sale_price = d_params["sale_price"]
        sale_limit = d_params["sale_limit"]
        sale_cost = d_params["sale_cost"]

        pv_area_ele = 0.15
        m = Model("Energy_Investment_Optimization")
        P_PV = m.addVar(vtype=GRB.CONTINUOUS, name="P_PV")
        P_ES = m.addVar(vtype=GRB.CONTINUOUS, name="P_ES")
        P_CG = m.addVar(vtype=GRB.CONTINUOUS, name="P_CG")
        IRR = m.addVar(vtype=GRB.CONTINUOUS, name="P_CG")

        E_PV = P_PV * sun_hrs_year * pv_eff
        E_CG = P_CG * self.work_hours_year * self.cg_eff

        Revenue = (E_PV + E_CG) * self.ele_price + (
                sale_price - sale_cost - self.ele_price) * (
                          E_PV + E_CG - self.load_year * self.work_hours_year)

        # 这个不对

        Investment_Cost = P_PV * pv_cost + P_ES * self.es_cost + P_CG * self.cg_cost

        m.setObjective(IRR, GRB.MINIMIZE)
        m.addConstr(IRR * Revenue == Investment_Cost)

        m.addConstr(P_PV <= min(pv_area_ele * pv_area, pv_limit),
                    "PV_Limit_Constraint")
        m.addConstr(P_CG * self.cg_eff >= self.imp_load_year, "CG_Constraint")
        m.addConstr(P_CG * self.cg_eff + P_PV * pv_eff >= max_load,
                    "CG_Min_Load_Constraint")
        m.addConstr(P_ES * self.es_eff >= 0.2 * pv_eff * P_PV,
                    "ES_PV_Constraint")
        m.addConstr(
            P_PV * pv_eff + P_ES * pv_eff + P_CG * self.cg_eff <= self.load_year + (
                    sale_limit / 24),
            "Total_Constraint")

        m.optimize()

        if m.status == GRB.OPTIMAL:
            irr = IRR.x
            if irr > self.max_invest_year:
                return {"Error": f"当前区域的投资回报比（IRR = {irr:.2f}）高于"
                                 f"设置的投资上限，建议不要投资。"}
            else:
                result = {
                    'pv': P_PV.x,
                    'es': P_ES.x,
                    'cg': P_CG.x,
                    'income_year': Revenue.getValue(),
                    'total_inv_cost': Investment_Cost.getValue(),
                    'irr': round(irr, 2)
                }
                return result
        else:
            return {"Error": "当前求解的限制条件不合理"}

    def com_strategy(self, d_params):
        pv_cost = d_params["pv_cost"]
        sun_hrs_year = d_params["sun_hrs_year"]
        max_load = d_params["max_load"]
        min_load = d_params["min_load"]
        pv_eff = d_params["pv_eff"]
        pv_area = d_params["pv_area"]
        pv_limit = d_params["pv_limit"]

        sale_price = d_params["sale_price"]
        sale_limit = d_params["sale_limit"]
        sale_cost = d_params["sale_cost"]

        pv_area_ele = 0.15
        m = Model("Energy_Investment_Optimization")
        P_PV = m.addVar(vtype=GRB.CONTINUOUS, name="P_PV")
        P_ES = m.addVar(vtype=GRB.CONTINUOUS, name="P_ES")
        P_CG = m.addVar(vtype=GRB.CONTINUOUS, name="P_CG")
        IRR = m.addVar(vtype=GRB.CONTINUOUS, name="P_CG")
        coefficient_of_combined_objective_1 = 0.5
        coefficient_of_combined_objective_2 = 0.4
        coefficient_of_combined_objective_3 = 0.1

        E_PV = P_PV * sun_hrs_year * pv_eff
        E_CG = P_CG * self.work_hours_year * self.cg_eff
        Revenue = (
                          self.ele_price * self.load_year * self.work_hours_year) + (
                          sale_price - sale_cost) * (
                          E_PV + E_CG - self.load_year * self.work_hours_year)
        Investment_Cost = P_PV * pv_cost + P_ES * self.es_cost + P_CG * self.cg_cost

        m.setObjective(IRR, GRB.MINIMIZE)
        m.addConstr(IRR * Revenue == Investment_Cost)
        m.addConstr(P_PV * pv_eff + E_CG * self.cg_eff >= max_load,
                    "PV_Constraint")
        m.addConstr(P_PV <= min(pv_area_ele * pv_area, pv_limit),
                    "PV_Limit_Constraint")
        m.addConstr(
            P_CG * self.cg_eff >= coefficient_of_combined_objective_1 * self.imp_load_year,
            "CG_Constraint")
        m.addConstr(P_CG * self.cg_eff + P_ES * self.es_eff >= 1.2 * min_load,
                    "CG_Min_Load_Constraint")
        m.addConstr(
            P_ES * self.es_eff >= coefficient_of_combined_objective_2 * pv_eff * P_PV,
            "ES_PV_Constraint")
        m.addConstr(
            P_PV * pv_eff + P_ES * pv_eff + P_CG * self.cg_eff <= self.load_year + (
                    sale_limit / 24),
            "Total_Constraint")

        m.optimize()

        if m.status == GRB.OPTIMAL:
            irr = Investment_Cost.getValue() / Revenue.getValue()
            if irr > self.max_invest_year:
                return {"提示": f"当前区域的投资回报比（IRR = {irr:.2f}）高于"
                                f"设置的投资上限，建议不要投资。"}
            else:
                result = {
                    'pv': P_PV.x,
                    'es': P_ES.x,
                    'cg': P_CG.x,
                    'income_year': Revenue.getValue(),
                    'total_inv_cost': Investment_Cost.getValue(),
                    'irr': round(irr, 2)
                }
                return result
        else:
            return {"Error": "当前求解的限制条件不合理"}


if __name__ == '__main__':
    load_year = 4500  # 年均负荷(kw)
    imp_load_year = 2000  # 年均重要负荷(kw)
    es_cost = 1500  # 储能成本  元/kw
    pv_cost = 800  # 光伏成本 元/kw
    ele_price = 1.4  # 电价
    cg_cost = 8000  # 煤制气成本加上燃料成本 元/kw
    sale_price = 1000  # 售电价格
    sale_cost = 0.5  # 售电成本
    work_hours_year = 2000  # 每年的工作时长为2000小时
    sun_hrs_year = 1300  # 每年的光照时长为1300小时
    max_load = 6000  # kw
    min_load = 2400  # kw
    pv_area = 50000  # 当前场地的光伏建设面积最大面积平方米
    pv_eff = 0.8  # 光伏效率
    max_invest_year = 5  # 当前投资收益最大限制
    pv_limit = 7000  # 光伏最大限制
    sale_limit = 100000  # 售电限制kw*h

    ems_model = EnergyModel(dict(load_year=load_year,
                                 imp_load_year=imp_load_year,
                                 es_cost=es_cost, ele_price=ele_price,
                                 cg_cost=cg_cost,
                                 work_hours_year=work_hours_year,
                                 max_invest_year=max_invest_year,
                                 cg_eff=0.8, es_eff=0.8))

    rlt_cri = ems_model.cri_calculate()
    print("rlt_cri", rlt_cri)
    rlt_all = ems_model.all_calculate(dict(pv_cost=pv_cost,
                                           sun_hrs_year=sun_hrs_year,
                                           max_load=max_load,
                                           min_load=min_load,
                                           pv_eff=pv_eff,
                                           pv_area=pv_area,
                                           pv_limit=pv_limit))
    print("rlt_all", rlt_all)

    rlt_sale = ems_model.sale_calculate(dict(pv_cost=pv_cost,
                                             sun_hrs_year=sun_hrs_year,
                                             max_load=max_load,
                                             min_load=min_load,
                                             pv_eff=pv_eff,
                                             pv_area=pv_area,
                                             pv_limit=pv_limit,
                                             sale_price=sale_price,
                                             sale_limit=sale_limit,
                                             sale_cost=sale_cost))
    print("rlt_sale", rlt_sale)

    rlt_com = ems_model.com_strategy(dict(pv_cost=pv_cost,
                                          sun_hrs_year=sun_hrs_year,
                                          max_load=max_load,
                                          min_load=min_load,
                                          pv_eff=pv_eff,
                                          pv_area=pv_area,
                                          pv_limit=pv_limit,
                                          sale_price=sale_price,
                                          sale_limit=sale_limit,
                                          sale_cost=sale_cost))
    print("rlt_com", rlt_com)
