# -*- coding:utf-8 -*-
"""用电健康各项偏差"""
# from pot_libs.logger import log


dev_ranges = {
    # 电压偏差
    "v": [
        [(80, 100), [(0, 0.05), (-0.05, 0)]],
        [(70, 80), [(0.05, 0.07), (-0.07, -0.05)]],
        [(60, 70), [(0.07, 0.1), (-0.1, -0.07)]],
        [(0, 60), [(0.1, 0.2), (-0.2, -0.1)]],
        [0, [(0.2, 1), (-1, -0.2)]]
    ],
    # 频率
    "freq": [
        [(90, 100), [(0, 0.05), (-0.05, 0)]],
        [(80, 90), [(0.05, 0.1), (-0.1, -0.05)]],
        [(60, 80), [(0.1, 0.2), (-0.2, -0.1)]],
        [(0, 60), [(0.2, 0.5), (-0.5, -0.2)]],
        [0, [(0.5, 1), (-1, -0.5)]]
    ],
    # 三相电压不平衡度
    "ubl": [
        [(90, 100), [(0, 0.005)]],
        [(80, 90), [(0.005, 0.01)]],
        [(60, 80), [(0.01, 0.02)]],
        [(0, 60), [(0.02, 0.04)]],
        [0, [(0.04, 1)]]
    ],
    # 功率因数
    "costtl": [
        [(90, 100), [(0.95, 1.0), (-1.0, -0.95)]],
        [(80, 90), [(0.9, 0.95), (-0.95, -0.9)]],
        [(60, 80), [(0.85, 0.9), (-0.9, -0.85)]],
        [(0, 60), [(0.65, 0.85), (-0.85, -0.65)]],
        [0, [(0, 0.65), (-0.65, 0)]]
    ],
    # 谐波畸变率
    "thdu": [
        [(90, 100), [(0, 0.02)]],
        [(80, 90), [(0.02, 0.03)]],
        [(60, 80), [(0.03, 0.05)]],
        [(0, 60), [(0.05, 0.1)]],
        [0, [(0.1, 1)]]
    ],
    # 负载率
    "lf": [
        [100, [(-1, 0.8)]],
        [60, [(0.8, 0.9)]],
        [0, [(0.9, 1)]]
    ],
    
}


def get_dev_grade(dev_type, cur):
    """获取分数档数等级"""
    if dev_type not in dev_ranges:
        # log.error("dev_type error")
        return None
    
    if cur is None or isinstance(cur, str):
        return None
    
    for _score, ranges in dev_ranges[dev_type]:
        for range in ranges:
            if range[0] == -1 and cur <= range[1]:
                return _score if isinstance(_score, int) else None
            
            if range[1] == 1 and cur >= range[0]:
                return _score if isinstance(_score, int) else _score[0]
            
            if cur >= range[0] and cur <= range[1]:
                return _score if isinstance(_score, int) else _score[0]
    
    # log.error("data error")
    return None


def get_dev_score(dev_type, cur):
    """"获取评分"""
    if dev_type not in dev_ranges:
        # log.error("dev_type error")
        return None

    for _score, ranges in dev_ranges[dev_type]:
        for range in ranges:

            if cur >= range[0] and cur <= range[1]:
                if isinstance(_score, int):
                    return _score

            if range[1] <= 0:
                if cur >= range[0] and cur <= range[1]:
                    if isinstance(_score, int):
                        return _score

                    ratio = (cur - range[0]) / (range[1] - range[0])
                    if dev_type in ["v", "freq", "ubl", "thdu", "lf"]:
                        # 负数区间，这个值成正比关系
                        return _score[0] + (_score[1] - _score[0]) * ratio
                    else:
                        return _score[1] - (_score[1] - _score[0]) * ratio

            if range[0] >= 0:
                if cur >= range[0] and cur <= range[1]:
                    if isinstance(_score, int):
                        return _score
                    ratio = (cur - range[0]) / (range[1] - range[0])
                    if dev_type in ["v", "freq", "ubl", "thdu", "lf"]:
                        # 正数区间，这个值成反比关系
                        return _score[1] - (_score[1] - _score[0]) * ratio
                    else:
                        return _score[0] + (_score[1] - _score[0]) * ratio

    return None


if __name__ == '__main__':
    res = get_dev_grade(dev_type="v", cur=0.01)
    print(res)

    res = get_dev_score(dev_type="thdu", cur=0.015)
    print(res)

    assert get_dev_score(dev_type="v", cur=0.01) == 96
    assert get_dev_score(dev_type="v", cur=-0.01) == 96

    assert get_dev_score(dev_type="v", cur=0.06) == 75
    assert get_dev_score(dev_type="v", cur=-0.06) == 75

    assert get_dev_score(dev_type="v", cur=0.08) == 70-10*(0.08-0.07)/(0.1-0.07)
    assert get_dev_score(dev_type="v", cur=-0.09) == 60 + 10*(0.01/0.03)

    assert get_dev_score(dev_type="v", cur=0.15) == 60 - 60 * ((0.15-0.1) / (0.2-0.1))
    assert get_dev_score(dev_type="v", cur=-0.15) == 0 + 60 * ((-0.15 + 0.2) / (0.2 - 0.1))

    assert get_dev_score(dev_type="v", cur=0.3) == 0
    assert get_dev_score(dev_type="v", cur=0.8) == 0
    assert get_dev_score(dev_type="v", cur=-0.2) == 0


    # 测试功率因数
    assert get_dev_score(dev_type="costtl", cur=0.98) == 90 + 10 * ((0.98-0.95)/(1-0.95))
    assert get_dev_score(dev_type="costtl", cur=-0.98) == 100 - 10 * ((-0.98+1)/(-0.95+1))

    assert get_dev_score(dev_type="costtl", cur=0.4) == 0
    assert get_dev_score(dev_type="costtl", cur=-0.4) == 0

