from dataclasses import dataclass
from pot_libs.sanic_api import Model
from pot_libs.sanic_api.column import List, Str, Int, Opt, Float
from pot_libs.common.components.fields import Cid, Item, DateTime
from unify_api.utils.response_code import DbErr, ParamErr


@dataclass
class FineMonitorChartReq(Model):
    '''
        精确监测-图表请求
    '''
    pid: int = Int("监测点id").eg(261)
    start: str = Str("开始时间").eg("2022-06-20 00:00:00")
    end: str = Str("结束时间").eg("2022-06-20 23:59:59")
    location_ids: list = List("location_ids").eg([487, 488, 489, 490, 491])


@dataclass
class FineMonitorInfoReq(Model):
    '''
        精确监测-指标统计请求
    '''
    pid: int = Int("监测点id").eg(261)
    start: str = Str("开始时间").eg("2022-06-20 00:00:00")
    end: str = Str("结束时间").eg("2022-06-20 23:59:59")
    location_ids: list = List("location_ids").eg([487, 488, 489, 490, 491])


@dataclass
class Chart(Model, DbErr):
    '''图表格式'''
    item: str = Opt(Str('类型').eg('A相'))
    value_slots: list = Opt(List('数据列表').items(Float()))


@dataclass
class Statistics(Model, DbErr):
    type: str = Str('指标类型').eg('temperature')
    item: Item = Item
    max: float = Float("最大值")
    max_time: DateTime = DateTime
    min: float = Float("最小值")
    min_time: DateTime = DateTime
    avg: float = Float('平均值')


@dataclass
class FineMonitorChartResp(Model, DbErr, ParamErr):
    '''
        精确监测-图表返回
    '''
    time_slots: list = List("横坐标时序").items(Float())
    residual_current: list = List("漏电流").items(Chart)
    temperature: list = List("温度").items(Chart)
    power: list = List("功率(p表示有功,q表示无功)").items(Chart)
    i: list = List("电流").items(Chart)
    v: list = List("电压").items(Chart)
    ctnum: int = Int("接线方式：包含两表法、三表法").eg(3)

    @classmethod
    def example(cls):
        return {
            "time_slots": ["00:00", "00:15"],
            "ctnum": 3,
            "residual_current": [{
                "item": "漏电流",
                "value_slots": [3.26, 3.31]
            }],
            "temperature": [{
                "item": "A相",
                "value_slots": [27.48, 27.43]
            }, {
                "item": "B相",
                "value_slots": [28.04, 28.03]
            }],
            "power": [{
                "item": "p",
                "value_slots": []
            }],
            "i": [{
                "item": "ia",
                "value_slots": [12, 34]
            }],
            "v": [{
                "item": "ua",
                "value_slots": []
            }]
        }


@dataclass
class FineMonitorInfoResp(Model, DbErr, ParamErr):
    '''
        精确监测-指标统计返回
    '''
    info_list: list = List("指标统计列表").items(Statistics)


@dataclass
class ElectricIndexParam(Model):
    item: str = Str("指标").eg("有功功率P(kW)")
    max: float = Float("最大值").eg(3081.60)
    max_time: str = Str("最大时间").eg("2022-12-24 23:59:59")
    min: float = Float("最小值").eg(324)
    min_time: str = Str("最小时间").eg("2022-12-24 12:12:12")
    avg: float = Float("平均值").eg(2311.12)


@dataclass
class ElectricIndexListReq(Model):
    cid: Cid
    mtid: int = Int("单个mtid").eg(1)
    start_time: str = Str("开始时间").eg("2022-12-24 00:00:00")
    end_time: str = Str("结束时间").eg("2022-12-24 23:59:59")

@dataclass
class ElectricIndexParam(Model):
    type: str = Str("类型").eg("temperature")
    item: str = Str("指标").eg("有功功率P(kW)")
    max: float = Float("最大值").eg(3081.60)
    max_time: str = Str("最大时间").eg("2022-12-24 23:59:59")
    min: float = Float("最小值").eg(324)
    min_time: str = Str("最小时间").eg("2022-12-24 12:12:12")
    avg: float = Float("平均值").eg(2311.12)


@dataclass
class ElectricIndexListResp(Model):
    ctnum: int = Int("接线方式 2-两表法 3-三表法").eg(1)
    general_param: list = List("常规参数").items(ElectricIndexParam)
    electric_quality: list = List("电能质量").items(ElectricIndexParam)
    safe_param: list = List("安全参数").items(ElectricIndexParam)
