from dataclasses import dataclass

from pot_libs.common.components.fields import Cid
from pot_libs.sanic_api import Model
from pot_libs.sanic_api.column import Opt, Float, Int, Str, List
from unify_api.utils.response_code import DbErr, ParamErr, JwtErr, UserErr


@dataclass
class HardwareInfoReq(Model):
    cid: Cid
    page_size: int = Opt(Int("頁大小").eg(10))
    page_num: int = Opt(Int("頁index").eg(1))


@dataclass
class HardwareInfo(Model, DbErr, ParamErr, JwtErr):
    """
    关于我们，硬件信息列表
    """

    installed_location: str = Opt(Str("安装位置").eg("3#变压器"))
    wiring_type: str = Opt(Str("接线形式").eg("三表法"))
    ct_change: str = Opt(Int("ct变化").eg(1000))
    pt_change: str = Opt(Int("pt变化").eg(1000))
    inline_capacity: int = Opt(Int("变压器容量").eg(1000))
    high_or_low_side: str = Opt(Str("高低压测").eg("高压侧or低压测"))
    rated_voltage: int = Opt(Int("额定电压").eg(400))


@dataclass
class HardwareInfoRespList(Model, DbErr, JwtErr, UserErr):
    rows: list = List("硬件信息").items(HardwareInfo)
    total: int = Int("总量")


@dataclass
class HardwareInfoCountReq(Model):
    cid: Cid


@dataclass
class HardwareInfoCountResp(Model):
    """
    关于我们硬件信息统计
    """

    installed_number: int = Int("安装点数").eg(1)
    legal_measurement_number: int = Int("法定计量点數量").eg(1)
    start_time: str = Str("启用时间").eg("2020-08-10 17:32")
    power_capacity: int = Int("供电容量").eg("4800")


@dataclass
class HisReq(Model):
    cid: Cid


@dataclass
class HisResp(Model):
    point_num: int = Int("监测户数").eg(48)
    device_num: int = Int("安装点数").eg(16)
    start_time: str = Str("启用时间").eg("2020-08-10 17:32")


@dataclass
class HlsReq(Model):
    cid: Cid
    page_size: int = Opt(Int("頁大小").eg(10))
    page_num: int = Opt(Int("頁index").eg(1))


@dataclass
class HlsResp(Model, DbErr, JwtErr, UserErr):
    rows: list = List("硬件信息").eg(
        {"room_name": "户号",
         "storey_name": "所属楼栋/楼层",
         "name": "安装位置",
         "sid": "设备编号",
         "create_time": "接入时间"
         })
    total: int = Int("总量")


@dataclass
class HardwareInfoManReq(Model):

    proxy_id: int = Opt(Int("代理id").eg(1))
    product: int = Int("product 1-知电 2-安电 3-安电管理 4-识电U").eg(3)


@dataclass
class HardwareInfoManResq(Model):
    project_count: int = Int("项目总数").eg(1)
    start_time: str = Str("启用时间").eg("2020-08-10 17:32")
    detection_point_count: int = Int("检测点数").eg(1)


@dataclass
class HardwareInfoListReq(Model):
    product: int = Int("product 1-知电 2-安电 3-安电管理 4-识电U").eg(3)
    proxy_id: int = Opt(Int("代理id").eg(1))
    page_size: int = Opt(Int("頁大小").eg(10))
    page_num: int = Opt(Int("頁index").eg(1))


@dataclass
class HardwareInfoList(Model, DbErr, ParamErr, JwtErr):
    """
    关于我们，硬件信息列表
    """

    cid: int = Opt(Str("项目id").eg(88))
    name: str = Opt(Str("项目名称").eg("公司本部502"))
    count_num: str = Opt(Int("监测点数").eg(5))
    create_time: str = Opt(Int("接入时间").eg("2020-08-10 17:32"))
    tcs_count: int = Opt(Int("供电容量").eg(1200))


@dataclass
class HardwareInfoListResq(Model):
    rows: list = List("项目检测点信息").items(HardwareInfoList)
    total: int = Int("总量")
